<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\MassDestroySpaceUnitRequest;
use App\Http\Requests\StoreSpaceUnitRequest;
use App\Http\Requests\UpdateSpaceUnitRequest;
use App\Models\SpaceUnit;
use App\Models\Tower;
use Gate;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;
use Yajra\DataTables\Facades\DataTables;

class SpaceUnitController extends Controller
{
    public function index(Request $request)
    {
        abort_if(Gate::denies('space_unit_access'), Response::HTTP_FORBIDDEN, '403 Forbidden');

        if ($request->ajax()) {
            $query = SpaceUnit::with(['tower'])->select(sprintf('%s.*', (new SpaceUnit())->table));
            $table = Datatables::of($query);

            $table->addColumn('placeholder', '&nbsp;');
            $table->addColumn('actions', '&nbsp;');

            $table->editColumn('actions', function ($row) {
                $viewGate = 'space_unit_show';
                $editGate = 'space_unit_edit';
                $deleteGate = 'space_unit_delete';
                $crudRoutePart = 'space-units';

                return view('partials.datatablesActions', compact(
                'viewGate',
                'editGate',
                'deleteGate',
                'crudRoutePart',
                'row'
            ));
            });

            $table->editColumn('id', function ($row) {
                return $row->id ? $row->id : '';
            });
            $table->addColumn('tower_name', function ($row) {
                return $row->tower ? $row->tower->name : '';
            });

            $table->editColumn('type', function ($row) {
                return $row->type ? SpaceUnit::TYPE_RADIO[$row->type] : '';
            });
            $table->editColumn('status', function ($row) {
                return $row->status ? SpaceUnit::STATUS_RADIO[$row->status] : '';
            });
            $table->editColumn('sensor', function ($row) {
                return $row->sensor ? $row->sensor : '';
            });
            $table->editColumn('lat', function ($row) {
                return $row->lat ? $row->lat : '';
            });
            $table->editColumn('long', function ($row) {
                return $row->long ? $row->long : '';
            });

            $table->rawColumns(['actions', 'placeholder', 'tower']);

            return $table->make(true);
        }

        return view('admin.spaceUnits.index');
    }

    public function create()
    {
        abort_if(Gate::denies('space_unit_create'), Response::HTTP_FORBIDDEN, '403 Forbidden');

        $towers = Tower::pluck('name', 'id')->prepend(trans('global.pleaseSelect'), '');

        return view('admin.spaceUnits.create', compact('towers'));
    }

    public function store(StoreSpaceUnitRequest $request)
    {
        $spaceUnit = SpaceUnit::create($request->all());

        return redirect()->route('admin.space-units.index');
    }

    public function edit(SpaceUnit $spaceUnit)
    {
        abort_if(Gate::denies('space_unit_edit'), Response::HTTP_FORBIDDEN, '403 Forbidden');

        $towers = Tower::pluck('name', 'id')->prepend(trans('global.pleaseSelect'), '');

        $spaceUnit->load('tower');

        return view('admin.spaceUnits.edit', compact('spaceUnit', 'towers'));
    }

    public function update(UpdateSpaceUnitRequest $request, SpaceUnit $spaceUnit)
    {
        $spaceUnit->update($request->all());

        return redirect()->route('admin.space-units.index');
    }

    public function show(SpaceUnit $spaceUnit)
    {
        abort_if(Gate::denies('space_unit_show'), Response::HTTP_FORBIDDEN, '403 Forbidden');

        $spaceUnit->load('tower');

        return view('admin.spaceUnits.show', compact('spaceUnit'));
    }

    public function destroy(SpaceUnit $spaceUnit)
    {
        abort_if(Gate::denies('space_unit_delete'), Response::HTTP_FORBIDDEN, '403 Forbidden');

        $spaceUnit->delete();

        return back();
    }

    public function massDestroy(MassDestroySpaceUnitRequest $request)
    {
        SpaceUnit::whereIn('id', request('ids'))->delete();

        return response(null, Response::HTTP_NO_CONTENT);
    }
}
